spool dbxml_wme.log

SET SCAN OFF
SET ECHO OFF
CREATE OR REPLACE PACKAGE dbxml_wme IS


-- *************************************************************************
-- ** This is a modification of the original Oracle-provided dbxml.sql package. This package was 
-- ** modified by S. Kho on August 4, 2003.
-- **
-- ** - Modified PROCEDURE XMLTagFor:- Removed double quotes for xml Tag Name.
-- ** - Modified FUNCTION ColumnsOfSelectStmt:- Removed uppercase function for column name.
-- ** - Modified PROCEDURE Get:- Removed uppercase function for variable (theSelectList).
-- *************************************************************************


  /*======================================================================
  ** Copyright (c) 1998 ORACLE Corporation
  **
  **     DBXML: Generates a Well-Formed XML Document from the results of
  **     ~~~~~  a query against an Oracle Database table.  The
  **            resulting XML document can automatically return
  **            associated data (via FK/PK constraints) to any desired
  **            level.
  **
  **     AUTHOR: Steve Muench, XML Technical Evangelist,
  **     ~~~~~~  Java Business Objects Dev Team
  **             Oracle Corporation.
  **
  **    CREDITS: I borrowed the heart of WebView's generic query parsing
  **    ~~~~~~~  code from BR_RPT.Simple_Query to form the basis of the
  **             dynamic SQL Fetching for the current table's row
  **             data. Thank you to Mike H., Chris B., and Tom K.  for
  **             writing WebView, the most clever bunch of PL/SQL I've
  **             seen yet!
  **
  ** DISCLAIMER: This is unsupported sample code which comes with 
  ** ~~~~~~~~~~  no warranties from Oracle Corporation or its author.
  **             The code is provided as a working example from which
  **             you might learn a thing or two about applying XML to
  **             Oracle Data using existing Oracle technologies like
  **             PL/SQL and the Oracle Web/App Server's PL/SQL Agent.
  **
  **   FEEDBACK: While I make no promises about fixing bugs, I am 
  **   ~~~~~~~~  interested in hearing from you if this example
  **             was helpful or if you have thought of some clever
  **             ways to extend it. Mail me at smuench@oracle.com
  **
  **      DOC: http://xml.us.oracle.com/dbxml/doc.html
  **      ~~~  (Oracle intranet only, sorry.)
  **
  **======================================================================
  **
  ** Date       Author  Version Comments
  ** =========  ======= ======= ==========================================
  ** 06/28/98   smuench 1.0.0   Created.
  ** 07/01/98   smuench 1.1.0   Lowercase <?xml, handle recursive tables
  **                            Support simple operators in query criteria
  ** 10/15/98   smuench 1.1.1   Cleaned up for distribution.
  ** 10/27/98   smuench 1.1.2   Fixed a bug for PL/SQL 8.0.5 strictness
  ** 11/09/98   smuench 1.1.3   Allow user-specified DocElt & Handle 0 Rows
  ** 11/11/98   smuench 1.1.4   For top table, allow user to specify
  **                              - SELECT Columns
  **                              - Order By Column List (COLNAME- for DESC)
  **                              - OR criteria instead of AND
  **                              - Case-Insensitive Query (COLNAME!)
  **                              - User-specified MaxRow count
  ** 11/16/98   smuench 1.1.5   Allow full SQL for top-level table
  **                            Quote common character entities in data values
  ** 11/20/98   smuench 1.1.6   Allow stylesheet name to be supplied
  ** 12/08/98   smuench 1.1.7   Beautify WHERE Clause Printing
  ** 01/12/99   smuench 1.1.8   Fix error in escaping less-than character
  ** 02/04/99   smuench 1.1.9   Allow override of Mime-type, default=text/xml
  **                            Fix select-list in iSQL for 7.X databases
  ** 02/21/99   smuench 1.1.10  Allow caller to specify tag for top-level table
  **
  **   ToDo List
  **   ------------------------------------------------------------------------
  **   - Hide foreign key colvalues in details
  **   - Support prototype DCD Schemas (or at least DT:DT per Element
  **
  ** Dependencies
  ** ~~~~~~~~~~~~
  **
  **   (*) PL/SQL Cartridge Runtime (OAS 2.x or 3.x)
  **
  **       -- Requires the OWA package for definition of VC_ARR type
  **       -- Calls HTP Package for Streaming XML to HTTP
  **
  ** Usage Notes
  ** ~~~~~~~~~~~
  **
  **   (0) Install this package in any schema you want to
  **       experiment with. You'll need to grant execute 
  **       access to this package to your OWA PL/SQL Agent
  **       schema owner.
  **
  **   (1) Call dbxml.Get() with the name of the table whose query
  **       results you want to format in XML. 
  **
  **   (2) Optionally provide initial WHERE clause criteria to
  **       limit the number of records from the starting table
  **       you pass into Get() by specifying an Array of column
  **       names. Ideally, you will provide criteria so that
  **       there is precisely one record from the starting table
  **       but Get() will work properly for multiple records, too.
  **
  **       When specified in an URL, repeating the 'kc=colname'
  **       parameter will add multiple WHERE Clause columns to
  **       the initial criteria list, and there must be a
  **       matching number of 'kv=value' parameters which will be
  **       used for the values in the WHERE Clause for the
  **       respective parameters. 
  ** 
  **       By default Get() will limit the top-level table results
  **       to a single row. This is to conform with the XML
  **       standard that the root node appear only once in the
  **       document. If you set the 'mult' parameter to 'Y', then
  **       Get() will allow multiple rows of results from the
  **       top level query and maintain conformity by adding a
  **       single enclosing <TableNameLIST> tag to enclose the
  **       (potentially) multiple rows of results.
  **
  **       An example URL which will return all of the employees
  **       in the EMP table whose 'JOB' is like '%MAN%' and whose
  **
  **       http://host/owa/dbxml.get?tb=emp&kc=job&kv=*man*
  **
  **       Note that the percent character is problematic in
  **       URL's so I'm using star for wildcard.       
  **
  **   (3) By default, Get() will traverse database constraints
  **       related to the current table, both its own foreign keys
  **       and any other tables foreign keys referencing the
  **       current table. Get() keeps track of which foreign keys
  **       its already visited to avoid infinite loops when it
  **       starts "walking" these relationships to "drill into"
  **       related data. If you do NOT want to print the related data
  **       for the starting table, then pass in a 'N' for the 'tr'
  **       parameter.
  **
  **   (4) For self-referencing tables, the 'rd' parameter provides
  **       new control on a table-by-table basis of whether you
  **       want to walk "up" the recursive relationship, or walk
  **       "down". You cannot walk both "up" and "down" a
  **       self-referencing relationship, and the default is to
  **       walk "down". For example, formatting the EMP table's
  **       data, when printing the row for "KING" walking down
  **       the relationship prints the list of employees which
  **       KING manages. Conversely, if formatting the EMP
  **       table's data for the "WARD" employee, walking up the
  **       relationship would step up through the "chain of
  **       command" and format the managerial chain of
  **       "WARD". The syntax for the 'rd' parameter is a
  **       comma-separated list of entries of the form:
  **     
  **            tablename2:[UP|DOWN],tablename2[UP|DOWN],etc.
  **
  **       so an example would be:
  **
  **            emp:down,part:up
  **
  **       This parameter only has an effect on tables which
  **       reference themselves, and only affects the
  **       interpretation of their self-referencing constraint.
  **       Listing any non-self-referencing tables in list is
  **       harmless, they are ignored.
  **
  **   (5) Get() calls itself recursively to format the
  **       the results of child data for the current table's
  **       current row. The only query criteria which you can
  **       affect is the initial set for the starting table. For all
  **       other levels below the inital table, the query criteria are
  **       automatically provided by the information derived from the
  **       constraints which link the tables.
  **
  **   (6) When Get() follows constraints is does so from the point of
  **       view of the current table. If Get() determines that a
  **       constraint can give rise to a collection of results on the
  **       other side of the relationship, it will by default add an
  **       extra level of containing XML tags for the collection. The
  **       canonical name it chooses for this container tag is
  **       <CurrentTableNameLIST> so, for example, if I were using
  **       Get() to print the DEPT table, when it goes to print the
  **       EMP records corresponding to the current department it will
  **       produce tags which look like:
  **
  **                <DEPT>
  **                  <DEPTNO>20</DEPTNO>
  **                  <DNAME>RESEARCH</DNAME>
  **                  <LOC>DALLAS</LOC>
  **                  <EMPLIST>
  **                    <EMP>
  **                      <EMPNO>7566</EMPNO>
  **                      <ENAME>JONES</ENAME>
  **                         :
  **                    </EMP>
  **                         :
  **                  </EMPLIST>
  **                         :
  **
  **       If you do not want the enclosing collection tags, then
  **       pass in 'N' for the parameter 'tg'.
  **
  **   (7) To exclude tables from the result, list their names
  **       in a comma-separated list and pass the list as the
  **       'ex' parameter to Get(). This provides a low-tech
  **       cut-point mechanism to prevent the constraint 
  **       walking algorithm from "finding" any table that
  **       is listed in the exclude list. Excluding the
  **       top-level starting table gives an error.
  **       view of the current table. If Get() determines that a
  */

  -------------------------------------------------------
  version               CONSTANT VARCHAR2(8)  := '1.1.10';
  -------------------------------------------------------

  empty_vc_arr  owa.vc_arr;

  PROCEDURE Get(
     tb         VARCHAR2,                   /* Current Table Name          */
     lv         NUMBER     := 0,            /* Current Indent Level        */
     mult       VARCHAR2   := 'N',          /* Multiple Top-Level Rows OK? */
     tg         VARCHAR2   := 'Y',          /* Tag Nested Collections?     */
     tr         VARCHAR2   := 'Y',          /* Traverse to Related Tables? */
     kc         owa.vc_arr := empty_vc_arr, /* WHERE Clause Column Names   */
     kv         owa.vc_arr := empty_vc_arr, /* WHERE Clause Column Values  */
     ex         VARCHAR2   := NULL,         /* List of cut-point tables    */
                                            /* to exclude from traversal   */
     rd         VARCHAR2   := NULL,         /* List of recurse direction   */
                                            /* hints.                      */
     debug      VARCHAR2   := 'N',          /* Show Debug Messages         */
     sl         VARCHAR2   := 'N',          /* Show Levels in Comments     */
     prevConstr owa.vc_arr := empty_vc_arr, /* Prev. Visited Constraints   */
     cl         VARCHAR2   := NULL,         /* Optional Column List        */
     de         VARCHAR2   := NULL,         /* Optional Document Elt Name  */
     tt         VARCHAR2   := NULL,         /* Optional top-level tag name */
     ord        VARCHAR2   := NULL,         /* Optional OrderBy Col List   */
     cond       VARCHAR2   := 'AND',        /* AND or OR where clause cols */
     maxr       NUMBER     := 200,          /* Maximum Rows to Retrieve    */
     isql       VARCHAR2   := NULL,         /* Initial Top-level SQL Stmt  */
     ss         VARCHAR2   := NULL,         /* User-Supplied Stylesheet    */
     mt         VARCHAR2   := 'text/xml'    /* Mime-type for return        */
  );

  /*
  ** Call Get using a full SQL Statement as the top-level query
  */
  PROCEDURE Query( theQuery            VARCHAR2,
                   theMainTable        VARCHAR2 := NULL,
                   theDocElement       VARCHAR2 := NULL,
                   tableElement        VARCHAR2 := NULL,
                   singleRow           VARCHAR2 := 'N',
                   includeDetails      VARCHAR2 := 'N',
                   detailExclusionList VARCHAR2 := NULL,
                   maximumRows         NUMBER   := 200,
                   styleSheet          VARCHAR2 := NULL,
                   MIMEType            VARCHAR2 := 'text/xml' );

END;
.
/
show errors

CREATE OR REPLACE PACKAGE BODY dbxml_wme IS

  /*======================================================================
  ** TYPE DEFINITIONS
  ** ~~~~~~~~~~~~~~~~
  **
  ** Record to hold constraint info for current table
  */
  TYPE ConstraintArrayRecord IS
       RECORD (
                 constraintName VARCHAR2(40),
                 colPosition    NUMBER,
                 colName        VARCHAR2(40),
                 colValue       VARCHAR2(4000),
                 targetTable    VARCHAR2(40),
                 targetColName  VARCHAR2(40),
                 direction      VARCHAR2(3)
              );

  /*
  ** Array of constraint info records
  */
  TYPE constraintArray IS TABLE OF ConstraintArrayRecord
                          INDEX BY BINARY_INTEGER;

  curConstraint ConstraintArrayRecord;
  TABSPACES    CONSTANT NUMBER      := 2;
  OPENCOMMENT  CONSTANT VARCHAR2(4) := '<!--';
  CLOSECOMMENT CONSTANT VARCHAR2(3) := '-->';

  /*======================================================================
  ** PRIVATE PACKAGE GLOBALS
  ** ~~~~~~~~~~~~~~~~~~~~~~~
  **
  */
  MIMETypeForXML        VARCHAR2(80);
  wroteXMLProlog        BOOLEAN := FALSE;
  specifiedSelectList   BOOLEAN;
  specifiedStylesheet    BOOLEAN;
  userSuppliedStylesheet VARCHAR2(400);
  userSuppliedSelectList VARCHAR2(400);
  userSuppliedTopTag    VARCHAR2(80);
  specifiedOrderBy      BOOLEAN;
  userSuppliedOrderBy   VARCHAR2(400);
  Or_Instead_of_And     BOOLEAN;
  g_Debug               BOOLEAN := FALSE;
  timesIntoGet          NUMBER  := 0;
  showLevel             BOOLEAN := TRUE;  /* Set to TRUE to see Levels */
  topLevelMultiple      BOOLEAN := FALSE;
  tagGroups             BOOLEAN;
  traverseRelatedTables BOOLEAN;
  initialTable          VARCHAR2(40);
  excludeList           VARCHAR2(2000);
  recurseHint           VARCHAR2(2000);

  /*======================================================================
  ** PRIVATE PACKAGE FUNCTIONS/PROCEDURES
  ** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  **
  */

  ------------------------------------------------------------------------

  /*
  ** ite -- If, Then, Else
  */
  FUNCTION Ite(
                tf  IN BOOLEAN,
                yes IN VARCHAR2,
                no  IN VARCHAR2 := NULL
              ) RETURN VARCHAR2
  IS
  BEGIN
    IF ( tf ) THEN
      RETURN yes;
    ELSE
      RETURN no;
    END IF;
  END Ite;

  ------------------------------------------------------------------------

  /*
  ** QuoteCommonEntities -- Substitute character entites for &, <, <, etc.
  */
  FUNCTION QuoteCommonEntities( theText VARCHAR2 )
  RETURN VARCHAR2 
  IS
  BEGIN
      RETURN REPLACE(REPLACE(REPLACE(theText,'&','&amp;'),'<','&lt;'),'>','&gt;');

  END QuoteCommonEntities;

  ------------------------------------------------------------------------

  /*
  ** WriteToStream -- Write text to the HTML output stream
  */
  PROCEDURE WriteToStream(
                       theText VARCHAR2
                     )
  IS
  BEGIN
      htp.p( theText );
  END WriteToStream;


  ------------------------------------------------------------------------

  /*
  ** WriteXMLPrologtoStream -- Write XML Prolog into the Stream if Necessary
  */
  PROCEDURE WriteXMLPrologToStream
  IS
  BEGIN
    /*
    ** Put the proper XML prolog in the stream
    */
    IF NOT WroteXMLProlog THEN
      IF lower(MIMETypeForXML) <> 'text/html' THEN
        owa_util.mime_header(MIMETypeForXML,TRUE);
      END IF;
      WriteToStream('<?xml version="1.0" ?>');
      IF specifiedStylesheet THEN
        WriteToStream('<?xml:stylesheet type="text/xsl" href="'||
                      UserSuppliedStylesheet||'" ?>');
      END IF;
      WroteXMLProlog := TRUE;
    END IF;
  END WriteXMLPrologToStream;

  ------------------------------------------------------------------------

  /*
  ** WriteErrorToStream -- Write an Error Message to the Stream
  */
  PROCEDURE WriteErrorToStream( theErrorMessage VARCHAR2, theQuery VARCHAR2 := NULL)
  IS
  BEGIN
    WriteXMLPrologToStream;
    WriteToStream('<DBXML>'||CHR(10)||
                  ite(theQuery IS NOT NULL,'  <QUERY>'||theQuery||'</QUERY>'||CHR(10))||
                  '  <ERROR>'||theErrorMessage||'</ERROR>'||CHR(10)||
                  '</DBXML>');
  END WriteErrorToStream;

  ------------------------------------------------------------------------

  /*
  ** XMLTagFor -- Prints XML tags for an element
  **
  **              If 'closing' is TRUE  => Closing tag
  **                           is FALSE => Opening tag
  **                           is NULL  => Empty tag
  **              If 'content' is provided, its included between
  **                           opening and closing tags.
  **
  **              In order to properly handle a NULL content
  **              being passed, we set the default value of
  **              content to be something (we hope) no one
  **              will ever pass in. A Ctrl-B character.
  */
  PROCEDURE XMLTagFor(
                       s         VARCHAR2 ,
                       the_level NUMBER,
                       closing   BOOLEAN  := FALSE,
                       content   VARCHAR2 := CHR(2)
                     )
  IS

-- Removed double quotes 8/4/03 S.Kho
	l_string VARCHAR2(255) := REPLACE(s, '"', '');
  BEGIN
    IF content = CHR(2) THEN
      WriteToStream(Ite(showLevel,OPENCOMMENT||
                    LPAD(the_level,2)|| /* ','||timesIntoGet|| */ CLOSECOMMENT)||
                    LPAD(' ',TABSPACES*(the_level-1),' ')||
                    '<'||Ite(closing,'/')||l_string||
                    Ite(closing IS NULL,'/')||'>');
    ELSE

      WriteToStream(Ite(showLevel,OPENCOMMENT||
                    LPAD(the_level,2)|| /* ','||timesIntoGet|| */ CLOSECOMMENT)||
                    LPAD(' ',TABSPACES*(the_level-1),' ')||
                    '<'||l_string||'>'||content||'</'||l_string||'>');
    END IF;
  END XMLTagFor;

  ------------------------------------------------------------------------

  /*
  ** Strip whitespace, optionally removing tags.
  */
  FUNCTION StripWS( v VARCHAR2 )
  RETURN VARCHAR2 IS
    result VARCHAR2(32000);
  BEGIN
    result := RTRIM(LTRIM(TRANSLATE(v,CHR(13)||CHR(8)||CHR(10),'   ')));
    WHILE (INSTR(result,'  ') > 0) LOOP
      result := REPLACE(result,'  ',' ');
    END LOOP;
    RETURN result;

  END;

  ------------------------------------------------------------------------

  /*
  ** WriteQueryToStream -- Write out a query to the output stream
  */
  PROCEDURE WriteQueryToStream( q VARCHAR2 )
  IS
    theQuery VARCHAR2(32000) := StripWS(q);
    selectList VARCHAR2(10000); 
    fromClause VARCHAR2(100);
    whereClause VARCHAR2(4000);
    idx  NUMBER;
    idx2 NUMBER;
    idx3 NUMBER;
  BEGIN
    idx := INSTR(UPPER(theQuery),' FROM ');
    selectList := SUBSTR(theQuery,1,idx - 1);
    idx2 := INSTR(UPPER(theQuery),' WHERE ');
    idx3 := INSTR(UPPER(theQuery),' ORDER BY ');
    IF idx2 = 0 THEN
      fromClause  := SUBSTR(theQuery,idx+1);
    ELSE
      fromClause  := SUBSTR(theQuery,idx+1,idx2-idx);
      IF idx3 = 0 THEN 
        whereClause := SUBSTR(theQuery,idx2+1);
      ELSE
        whereClause := SUBSTR(theQuery,idx2+1,idx3-(idx2+1));
      END IF;
    END IF;
    whereClause := REPLACE(whereClause,' AND ',CHR(10)||'   AND ');
    whereClause := REPLACE(whereClause,' OR ', CHR(10)||'    OR ');
    WriteToStream(OPENCOMMENT);
    WriteToStream(ite(specifiedSelectList,'SELECT '||
                  SUBSTR(userSuppliedSelectList,2,LENGTH(usersuppliedselectlist)-2),
                  'SELECT *'));
    WriteToStream('  '||fromClause);
    IF whereClause IS NOT NULL THEN
      WriteToStream(' '||whereClause);
    END IF;
    IF specifiedOrderBy THEN 
       WriteToStream(' '||' ORDER BY '||replace(usersuppliedOrderBy,'-',' DESC'));
    END IF;
    WriteToStream(CLOSECOMMENT);

  END;

  ------------------------------------------------------------------------

  /*
  ** DebugMessage -- Write text to output stream only if Debugging
  */
  PROCEDURE DebugMessage(
                          s     VARCHAR2,
                          force BOOLEAN := FALSE
                        )
  IS
  BEGIN
    IF g_Debug OR force THEN
      WriteToStream('DEBUG%> '|| s);
    END IF;
  END DebugMessage;

  ------------------------------------------------------------------------

  /*
  ** PrimaryKeyOf -- Return the primary key constraint name for a table
  */
  FUNCTION PrimaryKeyOf(
                         tabName VARCHAR2
                       ) RETURN  VARCHAR2
  IS

    CURSOR Get_Primary_Key_Constraint_Of( t VARCHAR2 )
        IS SELECT constraint_name
             FROM user_constraints
            WHERE table_name = UPPER(t)
              AND constraint_type = 'P'; /* Primary Key */

    pk_name VARCHAR2(40);

  BEGIN
    OPEN Get_Primary_Key_Constraint_Of( tabName );
    FETCH Get_Primary_Key_Constraint_Of INTO pk_name;
    CLOSE Get_Primary_Key_Constraint_Of;
    RETURN pk_name;
  END PrimaryKeyOf;

  ------------------------------------------------------------------------

  /*
  ** TableConstrainedBy -- Return the tablename associated with a
  **                       constraint.
  */
  FUNCTION TableConstrainedBy(
                               c      VARCHAR2
                             )
                               RETURN VARCHAR2
  IS

    CURSOR Get_Table_Constrained_By( c VARCHAR2 )
        IS SELECT table_name
             FROM user_constraints
            WHERE constraint_name = UPPER(c);

    table_name VARCHAR2(40);

  BEGIN
    OPEN Get_Table_Constrained_By( c );
    FETCH Get_Table_Constrained_By INTO table_name;
    CLOSE Get_Table_Constrained_By;
    RETURN table_name;
  END TableConstrainedBy;

  ------------------------------------------------------------------------

  /*
  ** ColumnsOfConstraint -- Return the columns & positions for a
  **                        constraint.
  */
  PROCEDURE ColumnsOfConstraint(
                                 c                VARCHAR2,
                                 colNames     OUT owa.vc_arr,
                                 colPositions OUT owa.vc_arr
                               )
  IS

    CURSOR Columns_Of_Constraint( c VARCHAR2 )
        IS SELECT column_name, to_char(position) position
             FROM user_cons_columns
            WHERE constraint_name = UPPER(c)
            ORDER BY position;

    idx        number := 0;

  BEGIN
    FOR tab IN Columns_Of_Constraint( c ) LOOP
      idx := idx + 1;
      colNames(idx) := tab.column_name;
      colPositions(idx) := tab.position;
    END LOOP;
  END ColumnsOfConstraint;

  ------------------------------------------------------------------------

  /*
  ** IsOneToOne -- Return TRUE if constraint is 1-to-1, FALSE if 1-to-many
  */
  FUNCTION IsOneToOne (
                        c      VARCHAR2
                      ) RETURN BOOLEAN
  IS
    cCols       owa.vc_arr;
    cPos        owa.vc_arr;
    pCols       owa.vc_arr;
    pkname      VARCHAR2(40);
    tName       VARCHAR2(40);
    pCols_Count NUMBER;
    cCols_Count NUMBER;
    same        BOOLEAN := TRUE;
    theConstr   VARCHAR2(44);
  BEGIN
    IF c LIKE 'IN:%' THEN
      theConstr := SUBSTR(c,4);
    ELSIF c LIKE 'OUT:%' THEN
      theConstr := SUBSTR(c,5);
    ELSE
      theConstr := c;
    END IF;

    tName := TableConstrainedBy( theConstr );
    pkName := PrimaryKeyOf( tName );
    ColumnsOfConstraint ( theConstr, cCols, cPos /* Discard */);
    ColumnsOfConstraint ( pkName, pCols, cPos /* Discard */);
    cCols_Count := NVL(cCols.COUNT,-1);
    pCols_Count := NVL(pCols.COUNT,-1);

    IF cCols_Count = pCols_Count THEN

      FOR j in 1..cCols_Count LOOP
        IF cCols(j) <> pCols(j) THEN
          same := FALSE;
          EXIT;
        END IF;
      END LOOP;

    ELSE

      same := FALSE;

    END IF;

    RETURN same;

  END IsOneToOne;

  ------------------------------------------------------------------------

  /*
  ** FirstTableOfSelectStmt -- Return name of first table/alias in FROM clause
  */
  FUNCTION FirstTableOfSelectStmt( iSQL VARCHAR2 ) 
  RETURN VARCHAR2 IS
    fromClause    VARCHAR2(2000);
    fromStartsAt  NATURAL;
    fromEndsAt    BINARY_INTEGER;
    colCount      NATURAL := 0;
    curTab        VARCHAR2(400);
    theTable      VARCHAR2(80);
    curChar       VARCHAR2(1);
    i             NATURAL;
    lenSQL        NATURAL := LENGTH(iSQL);
    lenFrom      NATURAL;
  BEGIN
    fromStartsAt  := INSTR(upper(iSQL),'FROM') + 5;
    fromEndsAt    := INSTR(upper(iSQL),'WHERE') - 1;
    IF fromEndsAt < 0 THEN
      fromEndsAt    := INSTR(upper(iSQL),'ORDER BY') - 1;
      IF fromEndsAt < 0 THEN
        fromEndsAt := lenSQL;
      END IF;
    END IF; 
    /*
    ** Change any whitespace to space characters and trim at the ends
    */
    fromClause := StripWS(SUBSTR(iSQL,fromStartsAt,fromEndsAt-fromStartsAt+1));

    IF fromClause IS NOT NULL THEN
      i       := 0;
      lenFrom := LENGTH(fromClause);

      WHILE ( i <= lenFrom ) LOOP
        i := i + 1;
        curChar := SUBSTR(fromClause,i,1);
        IF curChar = CHR(32) THEN
          curTab := NULL;
          LOOP
            i := i + 1;
            EXIT WHEN (i > lenFrom);
            curChar := SUBSTR(fromClause,i,1);
            EXIT WHEN (curChar <> CHR(32));
          END LOOP;
        END IF;
        IF curChar = ',' OR i = lenFrom THEN
           IF i = lenFrom THEN 
             curTab := curTab || curChar;
           END IF;
           theTable := UPPER(curTab);
           EXIT;
        ELSE
          curTab := curTab || curChar;
        END IF;
      END LOOP;
      RETURN theTable;
    END IF;
    RETURN NULL;
  END;

  ------------------------------------------------------------------------

  /*
  ** ColumnsOfSelectStmt -- Return array of columns for a select statement
  */
  FUNCTION ColumnsOfSelectStmt( iSQL VARCHAR2 ) 
  RETURN OWA.VC_ARR IS
    selectList     VARCHAR2(2000);
    selectStartsAt NATURAL;
    fromStartsAt   NATURAL;
    colCount       NATURAL := 0;
    curCol         VARCHAR2(400);
    theColumns     OWA.VC_ARR;
    curChar        VARCHAR2(1);
    i              NATURAL;
    lenSelList     NATURAL;
  BEGIN
    selectStartsAt := INSTR(upper(iSQL),'SELECT ') + 7;
    fromStartsAt   := INSTR(upper(iSQL),'FROM') - 1;

    /*
    ** Change any whitespace to space characters and trim at the ends
    */
    selectList := StripWS(SUBSTR(iSQL,selectStartsAt,fromStartsAt-selectStartsAt));

    IF selectList IS NOT NULL THEN
      i          := 0;
      lenSelList := LENGTH(selectList);
      WHILE ( i < lenSelList ) LOOP
        i := i + 1;
        curChar := SUBSTR(selectList,i,1);
        IF curChar = CHR(32) THEN
          curCol := NULL;
          LOOP
            i := i + 1;
            curChar := SUBSTR(selectList,i,1);
            IF curChar <> CHR(32) OR i > lenSelList THEN
              EXIT;
            END IF;
          END LOOP;
        END IF;
        IF curChar = '''' THEN
          LOOP
            i := i + 1;
            curChar := SUBSTR(selectList,i,1);
            IF curChar = '''' OR i > lenSelList THEN
              curcol := NULL;
              EXIT;
            END IF;
          END LOOP;
        ELSIF curChar = ',' OR i = lenSelList THEN
           IF i = lenSelList THEN
             curCol := curCol || curChar;
           END IF;
           colCount := colCount + 1;
-- ** Removed uppercase function 8/4/03 S.Kho
--          theColumns(colCount) := UPPER(curCol);
           theColumns(colCount) := curCol;
           curCol := NULL;
        ELSE
          curCol := curCol || curChar;
        END IF;
      END LOOP;
      RETURN theColumns;
    END IF;
    RETURN theColumns;
  END;

  ------------------------------------------------------------------------

  /*
  ** ColumnsOf -- Return array of columns for a table.
  */
  FUNCTION ColumnsOf(
                      t      VARCHAR2
                    ) RETURN owa.vc_arr
  IS

    CURSOR Columns_Of( t VARCHAR2 )
        IS SELECT column_name
             FROM user_tab_columns
            WHERE table_name = UPPER(t)
            ORDER BY column_id;

    column_list owa.vc_arr;
    idx        NUMBER := 0;
  BEGIN
    FOR tab IN Columns_Of( t ) LOOP
        idx := idx + 1;
        column_list(idx) := tab.column_name;
    END LOOP;
    RETURN column_list;
  END ColumnsOf;

  ------------------------------------------------------------------------

  /*
  ** CaseIns -- Returns the four parts of a WHERE clause to do
  **            case-insensitive query while still using indexes
  */
  FUNCTION caseins( s VARCHAR2 , n NUMBER ) RETURN VARCHAR2 IS

   wks VARCHAR2(4000);

   a VARCHAR2(1);
   b VARCHAR2(1);
   c VARCHAR2(1);
   d VARCHAR2(1);
   pct_at NATURAL;
   local_s VARCHAR2(100) := LTRIM(RTRIM(s,'%'),'%');
  begin
    pct_at := INSTR(local_s,'%');
    IF pct_at > 0 THEN
      local_s := SUBSTR(local_s,1,pct_at-1);
    END IF;
    if local_s is null THEN
      RETURN '%';
    END IF;
    if LENGTH(local_s) = 1 THEN
      a := UPPER(s);
      b := LOWER(s);

      if    n = 1 THEN RETURN '%'||a||'%'; 
      ELSIF n = 2 THEN RETURN '%'||b||'%';
      ELSIF n = 3 THEN RETURN '%';
      ELSIF n = 4 THEN RETURN '%';
      END IF;

    ELSE

      a := UPPER(SUBSTR(local_s,1,1));
      b := LOWER(SUBSTR(local_s,1,1));
      c := UPPER(SUBSTR(local_s,2,1));
      d := LOWER(SUBSTR(local_s,2,1));

      if    n = 1 THEN RETURN '%'||a||c||'%'; 
      ELSIF n = 2 THEN RETURN '%'||a||d||'%';
      ELSIF n = 3 THEN RETURN '%'||b||c||'%';
      ELSIF n = 4 THEN RETURN '%'||b||d||'%';
      END IF;

    END IF;
 
  END;

  ------------------------------------------------------------------------

  /*
  ** WhereClause -- Returns a properly formatted Where Clause given
  **                an array of columns and values
  */
  FUNCTION whereClause (
                         sourceCols owa.vc_arr,
                         targetCols owa.vc_arr
                       ) RETURN     VARCHAR2
  IS
    c VARCHAR2(32000);
    atLeastOne BOOLEAN := FALSE;
    thecol     VARCHAR2(400);
    theval     VARCHAR2(400);
    theoper    VARCHAR2(6);
    n1         VARCHAR2(5);
    n2         VARCHAR2(5);
    n3         VARCHAR2(5);
    n4         VARCHAR2(5);

    CaseInsensitiveQuery BOOLEAN;

  BEGIN
    FOR j IN 1..sourceCols.COUNT LOOP
      theOper := '=';
      thecol := NVL(LTRIM(RTRIM(sourceCols(j))),CHR(2));
      theval := NVL(LTRIM(RTRIM(targetCols(j))),CHR(2));
      IF thecol <> CHR(2) AND theval <> CHR(2) THEN
        /*
        ** Handles some of the basic query operators.
        ** If the value has a percent, do a LIKE query, 
        ** otherwise check for inequalities, otherwise
        ** assume an equality.
        ** 
        ** If the query begins with a ^-sign, copy the
        ** rest of the string verbatim into the query.
        **
        ** 
        ** If theVal is still '###' then that's an error
        ** condition. It means the parent query did not
        ** include the column name to be used for the join
        ** so we can't do the join. We force the query
        ** for the details to return no rows by adding
        ** a 1=0 predicate.
        */
        IF theval = '###' THEN

          theOper := '=';
          theVal  := theCol|| ' AND 1=0';

        ELSIF theval LIKE '^%' THEN

          theOper := NULL;
          theVal  := REPLACE(theVal,'^');            

        ELSE

          IF INSTR(theval,'*') > 0 THEN

            theOper := 'LIKE';
            theVal  := REPLACE(theVal,'*','%');

          ELSE

            IF theval LIKE '<>%' THEN

              theOper := '<>';

            ELSIF theval LIKE '>=%' THEN

              theOper := '>=';

            ELSIF theval LIKE '<=%' THEN

              theOper := '<=';

            ELSIF theval LIKE '>%' THEN

              theOper := '>';

            ELSIF theval LIKE '<%' THEN

              theOper := '<';

            ELSIF theval LIKE '=%' THEN

              theOper := '=';

            END IF;

            /* 
            ** Strip the operator from the value
            */
            theVal  := REPLACE(theVal,theOper);

          END IF;
          
          /*
          ** If the column name ends with an
          ** exclamation point, then do the
          ** query case-INsensitively
          */
          IF theCol LIKE '%!' THEN
            n1 := caseins(theVal,1);
            n2 := caseins(theVal,2);
            n3 := caseins(theVal,3);
            n4 := caseins(theVal,4);
            theCol := SUBSTR(theCol,1,length(theCol)-1);
            CaseInsensitiveQuery := TRUE;
          ELSE
            CaseInsensitiveQuery := FALSE;
          END IF;

          theVal := ''''||theVal||''' ';

        END IF;

        IF CaseInsensitiveQuery THEN

          c := c||Ite(atleastOne,CHR(10)||'   AND ')||'(UPPER('||thecol||') '||theOper||' UPPER('||theval||') AND '||
               '('||thecol||' LIKE '''||n1||''' OR '||
               thecol||' LIKE '''||n2||''' OR '||
               thecol||' LIKE '''||n3||''' OR '||
               thecol||' LIKE '''||n4||'''))';

        ELSE

          c := c||Ite(atLeastOne,CHR(10)||'   AND ')||thecol||' '||theOper||' '||theval;

        END IF;
        atLeastOne := TRUE;
      END IF;
    END LOOP;
    RETURN Ite(c IS NOT NULL,'WHERE '||c);
  END WhereClause;


  ------------------------------------------------------------------------

  /*
  ** SelectList -- Returns a properly formatted Select list given
  **               an array of columns names
  */
  FUNCTION selectList (
                        listOfColumns owa.vc_arr
                      ) RETURN        VARCHAR2
  IS
    c VARCHAR2(32000);
  BEGIN
    FOR j IN 1..listOfColumns.COUNT LOOP
      c := c||Ite(j>1,', ')||listOfColumns(j);
    END LOOP;
    RETURN NVL(c,'*');
  END SelectList;

  /*======================================================================
  ** PUBLIC PACKAGE FUNCTIONS/PROCEDURES
  ** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  **
  */

  ------------------------------------------------------------------------

  /*
  ** Get - Gets and Formats a row of a Query in XML, including
  **       nested substructure.
  **
  **       Get "walks" the foreign key constraints which the current
  **       table has with other tables, and walks other tables'
  **       foreign key links to the current table. Then, it follows
  **       those links, and asks them to GET() there structure as
  **       subordinates of the current row of the current table.
  **
  **       When GET is done, the current ROW (and all its
  **       substructures) will be in XML syntax, and returned to the
  **       requestor.
  */
  PROCEDURE Get(
                 tb         VARCHAR2,                   /* Current Table Name          */
                 lv         NUMBER     := 0,            /* Current Indent Level        */
                 mult       VARCHAR2   := 'N',          /* Multiple Top-Level Rows OK? */
                 tg         VARCHAR2   := 'Y',          /* Tag Nested Collections?     */
                 tr         VARCHAR2   := 'Y',          /* Traverse to Related Tables? */
                 kc         owa.vc_arr := empty_vc_arr, /* WHERE Clause Column Names   */
                 kv         owa.vc_arr := empty_vc_arr, /* WHERE Clause Column Values  */
                 ex         VARCHAR2   := NULL,         /* List of cut-point tables    */
                                                        /* to exclude from traversal   */
                 rd         VARCHAR2   := NULL,         /* List of recurse direction   */
                                                        /* hints.                      */
                 debug      VARCHAR2   := 'N',          /* Show Debug Messages         */
                 sl         VARCHAR2   := 'N',          /* Show Levels in Comments     */
                 prevConstr owa.vc_arr := empty_vc_arr, /* Prev. Visited Constraints   */
                 cl         VARCHAR2   := NULL,         /* Optional Column List        */
                 de         VARCHAR2   := NULL,         /* Optional Document Elt Name  */
                 tt         VARCHAR2   := NULL,         /* Optional top-level tag name */
                 ord        VARCHAR2   := NULL,         /* Optional OrderBy Col List   */
                 cond       VARCHAR2   := 'AND',        /* AND or OR where clause cols */
                 maxr       NUMBER     := 200,          /* Maximum Rows to Retrieve    */
                 isql       VARCHAR2   := NULL,         /* Initial Top-level SQL Stmt  */
                 ss         VARCHAR2   := NULL,         /* User-Supplied Stylesheet    */
                 mt         VARCHAR2   := 'text/xml' )  /* Mime-type for return        */
  IS

    foundARow       BOOLEAN := FALSE;
    constrsSoFar    owa.vc_arr := prevConstr;
    atLeastOneRow   BOOLEAN := FALSE;
    cArr            ConstraintArray;
    tabArr          owa.vc_arr;
    consArr         owa.vc_arr;
    onemanyArr      owa.vc_arr;
    colNames        owa.vc_arr;
    the_level       NUMBER := lv;
    t               VARCHAR2(40) := UPPER(tb);
    q               VARCHAR2(32000);
    newColList      owa.vc_arr;
    newValList      owa.vc_arr;
    targetTableName VARCHAR2(44);
    oneormany       VARCHAR2(1);
    lastConstPtr    NUMBER := 0;
    columnValue     VARCHAR2(32767);
    theValue        VARCHAR2(32767);
    colCnt          NUMBER :=0;
    rowCnt          NUMBER:=0;
    theCursor       NUMBER;
    status          NUMBER;
    maxRows         INT := 0;
    TagName         VARCHAR2(200);
    max_rows        NUMBER := maxr;
    minimum_row     NUMBER :=1;
    thecolname      VARCHAR2(40);

    /*====================================================================
    ** Get() PRIVATE FUNCTIONS/PROCEDURES
    ** ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    **
    */

    --[ Get() ]-----------------------------------------------------------

    /*
    ** AlreadyVisited -- Returns TRUE if we've already visited the
    **                   constraint name passed in.
    */
    FUNCTION AlreadyVisited(
                             constraintName VARCHAR2
                           ) RETURN         BOOLEAN
    IS
    BEGIN
      FOR j IN 1..constrsSoFar.COUNT LOOP
        IF constrsSoFar(j) = UPPER(constraintName) THEN
          RETURN TRUE;
        END IF;
      END LOOP;
      RETURN FALSE;
    END AlreadyVisited;

    --[ Get() ]-----------------------------------------------------------

    /*
    ** MarkConstraintAsVisited -- Note the fact that we've seen the
    **                            constraint passed in.
    */
    PROCEDURE MarkConstraintAsVisited(
                                       constraintName VARCHAR2
                                     )
    IS
    BEGIN
      /* 
      **
      ** For self-referencing tables (e.g. EMP) the same
      ** constraint (e.g. EMP_SELF_KEY) will show up both as an
      ** Inbound and as an Outbound key during our walking. I
      ** modified the queries in the AddInboundConstraints() and
      ** AddOutboundConstraint to prepend the string 'IN:'
      ** before the inbound occurrence of the self-referencing
      ** key, and prepend 'OUT:' before the outbound occurrence
      ** of the self-referencing key so we can tell them apart. 
      **
      ** In order to allow recursion yet prevent strange
      ** looping, we need to adopt the following convention:
      **
      **  - If we want to walk "down" the tree of recursively
      **    referenced elements, this means we want to follow
      **    the 'IN:'bound constraint as many times "deep" as
      **    required, but never "look back" and follow the
      **    matching 'OUT:'bound key.
      **
      **  - If we want to walk "up" the tree of recursively
      **    referenced elements, this means we want to follow
      **    the 'OUT:'bound constraint as many times "up" as
      **    required, but never "look back down" and follow the
      **    matching 'IN:'bound key.
      **    
      ** So, I modified the queries which find constraints
      ** to block out finding the IN: or the OUT: bound
      ** based on a recurseHint string which can contain
      ** a comma-separated list of hints like: 
      **
      **    EMP:DOWN,PART:UP
      **
      ** which will properly leave out the direction
      ** which we don't want to encounter for the
      ** particular table at hand.
      **
      ** All this means that for a self-referencing 
      ** key, we should only ever find either the IN:
      ** or the OUT: but not both. Which one we define
      ** depends on the recurseHint with the default
      ** being that we prefer to find the IN: which 
      ** walks "down" the tree.
      ** 
      ** To allow recursion, when someone asks us
      ** to mark an IN: or OUT: constraint as visited,
      ** we just ignore the request and don't mark it.
      */
      IF     constraintName NOT LIKE 'IN:%' 
         AND constraintName NOT LIKE 'OUT:%'
      THEN

        constrsSoFar( constrsSoFar.COUNT + 1 ) := constraintName;

      END IF;



    END MarkConstraintAsVisited;

    --[ Get() ]-----------------------------------------------------------

    /*
    ** addInboundConstraintsFor -- Fill the constraint array with the
    **                             info for constraints which
    **                             are Foreign Keys to the table whose
    **                             name is passed in.
    */
    PROCEDURE addInboundFKConstraintsFor(
                                      tabName              VARCHAR2,
                                      constraintArr IN OUT ConstraintArray
                                      )
    IS

      CURSOR targetWCforConstraintsOfTable( t VARCHAR2 )
          IS SELECT
                    DECODE(ucc2.table_name,
                           t,
                           'IN:',
                           NULL)||
                uc.constraint_name constraint_name, /* Constraint Name */
                utc.column_id      column_id,       /* Col pos         */
                utc.column_name    source_column_name, /* Col Name    */
                '###'              theValue,        /* Col Val         */
                ucc2.table_name    table_name,      /* Target Table    */
                ucc2.column_name   column_name,     /* Target Column   */
                'IN'               direction        /* INbound,OUTbound*/
             FROM  user_tab_columns  utc,
                   user_cons_columns ucc,
                   user_constraints  uc,
                   user_cons_columns ucc2
             WHERE (         utc.table_name, utc.column_name)
                IN (
                      SELECT table_name    , column_name
                       FROM user_cons_columns
                      WHERE table_name = UPPER(t)
                   )
                   AND ucc.table_name       = utc.table_name
                   AND ucc.column_name      = utc.column_name
                   AND  uc.constraint_type  = 'R'
                   AND ucc.constraint_name  = uc.r_constraint_name
                   AND ucc2.constraint_name = uc.constraint_name
                   AND ucc2.position        = ucc.position
                   AND ucc2.owner           = uc.owner
                   AND INSTR(excludeList,','||ucc2.table_name||',') = 0
                   AND (
                          (
                            ucc2.table_name <> t
                          )
                          OR
                          (
                            (
                              ucc2.table_name = t
                            )
                            AND
                            (
                              ( /* Neither tabname:UP nor tabname:DOWN is in RecurseHint*/
                                   INSTR(recurseHint,','||ucc2.table_name||':UP,') = 0
                                AND INSTR(recurseHint,','||ucc2.table_name||':DOWN,') = 0
                              ) 
                              OR
                              ( /* tabname:DOWN is in the recurseHint */
                                INSTR(recurseHint,','||ucc2.table_name||':DOWN,') > 0
                              )
                            )
                          )
                        )
                   ORDER BY 1;

      idx                 NUMBER       := constraintArr.COUNT;
      lastConstraintWeSaw VARCHAR2(44) := '##';
      alreadySeenIt       BOOLEAN      := FALSE;
    BEGIN

      FOR j IN targetWCforConstraintsOfTable( UPPER(tabName) ) LOOP

        /*
        ** if this is the first row of constraint columns
        ** coming in for a new constraint name from
        ** the query (which is ordered by constraint name)
        */
        IF j.constraint_name <> lastConstraintWeSaw THEN

          IF lastConstraintWeSaw <> '##' AND NOT alreadySeenIt THEN

            MarkConstraintAsVisited( lastConstraintWeSaw );

          END IF;

          lastConstraintWeSaw := j.constraint_name;

          /*
          ** See if we've already traversed this constraint
          */
          alreadySeenIt := AlreadyVisited( j.constraint_name );

        END IF;

        IF NOT alreadySeenIt THEN
          idx := idx + 1;
          /*
          ** PL/SQL BUG in 8.0.4 (Seems related to Bug 752393)
          **
          ** constraintArr(idx) := j;
          */
          constraintArr(idx).constraintName := j.constraint_Name;
          constraintArr(idx).colPosition    := j.column_id;
          constraintArr(idx).colName        := j.source_column_name;
          constraintArr(idx).colValue       := j.theValue;
          constraintArr(idx).targetTable    := j.table_name;
          constraintArr(idx).targetColName  := j.column_name;
          constraintArr(idx).direction      := j.direction;

        END IF;

      END LOOP;

      IF lastConstraintWeSaw <> '##' THEN
        MarkConstraintAsVisited( lastConstraintWeSaw );
      END IF;
 
    END addInboundFKConstraintsFor;

    --[ Get() ]-----------------------------------------------------------

    /*
    ** addOutboundConstraintsFor -- Fill the constraint array with the
    **                              info for constraints which
    **                              are the Foreign Keys of the table whose
    **                              name is passed in.
    */
    PROCEDURE addOutboundFKConstraintsFor(
                                       tabName VARCHAR2,
                                       constraintArr IN OUT ConstraintArray
                                       )
    IS

      CURSOR xyz2 ( zzz3 VARCHAR2 )
          IS SELECT
                    DECODE(ucc2.table_name,
                           zzz3,
                           'OUT:',
                           NULL)||
                    uc.constraint_name constraint_name, /* Outbound Const */
                    utc.column_id      column_id,       /* Col Position   */
                    utc.column_name    source_column_name, /* Col Name    */
                    '###'              theValue,        /* Col Value      */
                    ucc2.table_name    table_name,      /* Target Table   */
                    ucc2.column_name   column_name,     /* Target Column  */
                    'OUT'              direction        /* INbound,OUT    */
             FROM  user_tab_columns  utc,
                   user_cons_columns ucc,
                   user_constraints  uc,
                   user_cons_columns ucc2
             WHERE
                   uc.constraint_type   = 'R'
               AND uc.table_name        = UPPER(zzz3)
               AND uc.constraint_name   = ucc.constraint_name
                                          /* The name of the outbound FK */
               AND INSTR(excludeList,','||ucc2.table_name||',') = 0
               AND (
                      (
                        ucc2.table_name <> UPPER(zzz3)
                      )
                      OR
                      (
                        (
                          ucc2.table_name = UPPER(zzz3)
                        )
                        AND
                        (
                          /* tabname:UP is in the recurseHint */
                          INSTR(recurseHint,','||ucc2.table_name||':UP,') > 0
                        )
                      )
                    )
               AND uc.owner             = ucc.owner
               AND ucc.table_name       = utc.table_name
               AND ucc.column_name      = utc.column_name
               AND uc.r_owner           = ucc2.owner
               AND uc.r_constraint_name = ucc2.constraint_name
                                          /* the name of PK refd by FK   */
             ORDER BY 1;

      idx                 NUMBER       := constraintArr.COUNT;
      lastConstraintWeSaw VARCHAR2(44) := '##';
      theTableName        VARCHAR2(40) := UPPER(tabname);
      alreadySeenIt       BOOLEAN      := FALSE;

    BEGIN

      FOR m IN xyz2( UPPER(theTableName) ) LOOP

        /*
        ** if this is the first row of constraint columns
        ** coming in for a new constraint name from
        ** the query (which is ordered by constraint name)
        */
        IF m.constraint_Name <> lastConstraintWeSaw THEN

          IF lastConstraintWeSaw <> '##' AND NOT alreadySeenIt THEN

            MarkConstraintAsVisited( lastConstraintWeSaw );

          END IF;

          lastConstraintWeSaw := m.constraint_Name;

          /*
          ** See if we've already traversed this constraint
          */
          alreadySeenIt := AlreadyVisited( m.constraint_Name );

        END IF;

        IF NOT alreadySeenIt THEN

          idx := idx + 1;
          /*
          ** PL/SQL BUG in 8.0.4 (Seems related to Bug 752393)
          **
          ** constraintArr(idx) := m;
          */
          constraintArr(idx).constraintName := m.constraint_Name;
          constraintArr(idx).colPosition    := m.column_id;
          constraintArr(idx).colName        := m.source_column_name;
          constraintArr(idx).colValue       := m.theValue;
          constraintArr(idx).targetTable    := m.table_name;
          constraintArr(idx).targetColName  := m.column_name;
          constraintArr(idx).direction      := m.direction;
        END IF;
      END LOOP;

      IF lastConstraintWeSaw <> '##' THEN
        MarkConstraintAsVisited( lastConstraintWeSaw );
      END IF;

    END addOutboundFKConstraintsFor;

    --[ Get() ]-----------------------------------------------------------

    /*
    ** UpdateConstraintJoinValues -- Sets the current rows values
    **                               in the array of info used to
    **                               produce the proper WHERE clause
    **                               for nested detail information.
    */
    PROCEDURE UpdateConstraintJoinValues(
                                     theColPos     NUMBER,
                                     theValue      VARCHAR2,
                                     constraintArr IN OUT ConstraintArray
                                     )
    IS
    BEGIN
      FOR i IN 1..constraintArr.COUNT LOOP
        IF constraintArr(i).colPosition = thecolPos THEN
          constraintArr(i).colValue := theValue;
        END IF;
      END LOOP;
    END UpdateConstraintJoinValues;

    --[ Get() ]-----------------------------------------------------------

    /*
    ** UpdateConstraintJoinValues -- Sets the current rows values
    **                               in the array of info used to
    **                               produce the proper WHERE clause
    **                               for nested detail information.
    */
    PROCEDURE UpdateConstraintJoinValues(
                                     theColName    VARCHAR2,
                                     theValue      VARCHAR2,
                                     constraintArr IN OUT ConstraintArray
                                     )
    IS
    BEGIN
      FOR i IN 1..constraintArr.COUNT LOOP
        IF constraintArr(i).colName = theColName THEN
          constraintArr(i).colValue := theValue;
        END IF;
      END LOOP;
    END UpdateConstraintJoinValues;

    --[ Get() ]-----------------------------------------------------------

    /*
    ** nextConstraint -- Returns the info for the next constraint
    **                   in the constraint array. The info might
    **                   be multi-record if the constraint is for
    **                   multiple columns
    */
    PROCEDURE nextConstraint(
                              constraintArr     IN     ConstraintArray,
                              lastconstraintpos IN OUT NUMBER,
                              tablename            OUT VARCHAR2,
                              oneOrMany            OUT VARCHAR2,
                              colNames             OUT owa.vc_arr,
                              colValues            OUT owa.vc_arr
                            )
    IS
      timesthrough   NUMBER := 0;
      x              NUMBER := lastconstraintpos;
      lastone        NUMBER := constraintArr.LAST;
      y              NUMBER := 0;
      l              VARCHAR2(44);
      constraintname VARCHAR2(44);
      constraintdir  VARCHAR2(3);
    BEGIN

      /*
      ** Make sure there are some constraint records
      ** in the constraint array. If not, just return.
      */
      IF constraintArr.COUNT = 0 OR lastconstraintpos >= lastone THEN

        RETURN; -- This will leave tableName null

      END IF;

      /*
      ** "peek" ahead to get the first constraint name
      ** which we'll point to in the loop
      */
      l := constraintArr( x + 1 ).constraintName;

      WHILE x < lastone LOOP
       timesthrough := timesthrough + 1;

       IF timesthrough > 50 THEN
         -----------------
         DebugMessage('Cool... An infinite loop!');
         -----------------
         RETURN;
       END IF;

        x := x + 1;

        IF constraintArr( x ).constraintName = l THEN
          y := y + 1;
          tableName      := constraintArr( x ).targetTable;
          constraintName := constraintArr( x ).constraintname;
          constraintDir  := constraintArr( x ).direction;
          colNames(y)    := constraintArr( x ).targetcolName;
          colValues(y)   := constraintArr( x ).colValue;
        ELSE
          x := x - 1;
          EXIT;
        END IF;

      END LOOP;

      lastconstraintpos := x;

      IF constraintdir = 'IN' THEN

        IF IsOneToOne( constraintname ) THEN
          oneormany := '1';
        ELSE
          oneormany := 'M';
        END IF;
      ELSE
        oneormany := 1;
      END IF;

    EXCEPTION
       WHEN OTHERS THEN
         WriteToStream('ERROR IN NEXTCONSTRAINT: '||SQLERRM||', x='||x);
         RAISE;
    END nextConstraint;


  /*
  ** Get() Main Code
  **
  */
  BEGIN

    timesIntoGet := timesIntoGet + 1;

    /*
    ** This stuff needs to run ONLY the first time through.
    */
    IF the_level = 0 THEN

      /*
      ** Process State Variables
      **
      **             tagGroups (tg) defaults to 'Y'
      ** traverseRelatedTables (tr) defaults to 'Y'
      */
      WroteXMLProlog         := FALSE;
      MIMETypeForXML         := mt;
      Or_Instead_Of_And      := upper(cond) = 'OR';
      specifiedSelectList    := cl is not null;
      userSuppliedSelectList := ','||upper(cl)||',';
      specifiedStylesheet    := replace(UPPER(ss),'NONE') is not null;
      userSuppliedStylesheet := ss;
      specifiedOrderby       := ord is not null;
      userSuppliedOrderBy    := upper(ord);
      userSuppliedTopTag     := upper(tt);
      g_Debug                := NVL(UPPER(debug),'Y') = 'Y';
      showLevel              := NVL(UPPER(sl),'N') = 'Y';
      initialTable           := t;
      topLevelMultiple       := NVL(UPPER(mult),'N')  = 'Y';
      tagGroups              := NVL(UPPER(SUBSTR(tg,1,1)),'Y') ='Y';
      traverseRelatedTables  := NVL(UPPER(SUBSTR(tr,1,1)),'Y') ='Y';
      excludeList            := ','||UPPER(ex)||',';
      recurseHint            := ','||UPPER(rd)||',';

      /*
      ** Do Some initial error checking...
      */

      IF tb IS NULL THEN
	/*
	** If initial table is in the Exclude list, then
	** there's nothing to do
	*/
        WriteErrorToStream('You Forgot to Specify a Table.');
        RETURN;  
      ELSIF INSTR(excludeList, ','||initialTable||',') > 0 THEN
	/*
	** If initial table is in the Exclude list, then
	** there's nothing to do
	*/
        WriteErrorToStream('You Asked to Exclude the Initial Table.');
        RETURN;
      END IF;

      
      /*
      ** Put the proper XML prolog in the stream
      */
      WriteXMLPrologToStream;

      /*
      ** Not clear if we can write in a
      ** DOCTYPE without providing a DTD.
      ** Maybe that doesn't even make sense.
      ** For now, leave it out.
      **
      ** WriteToStream('<!DOCTYPE '||t||Ite(topLevelMultiple,'LIST')||'>');
      */
      WriteToStream('<!-- Oracle DBXML Version '||version||' Query Results at '||
                     TO_CHAR(SYSDATE,'DD-MON-YYYY HH24:MI:SS')||' -->');

    END IF;

    IF the_level = 0 AND NOT topLevelMultiple THEN
      max_rows := 1;
    END IF;

    /*
    ** Populate the constraint Array for this table
    **
    ** The constraint array contains information on
    ** constraints which refer to the current table
    ** and constraints which the current table has
    ** which refer to other tables. These are
    ** called "Inbound" and "Outbound" respectively.
    **
    ** The AddInbound... and AddOutbound... routines
    ** keep track of which constraints we've already
    ** visited and make sure to not traverse the
    ** same constraint name twice.
    */

    addInboundFKConstraintsFor( t,   /* current table name            */
                                cArr /* This table's constraint array */
                              );

    addOutboundFKConstraintsFor( t,  /* current table name            */
                                cArr /* This table's constraint array */
                              );

    /*
    ** Get the Columns of the current table
    */
    IF timesIntoGet = 1 AND iSQL IS NOT NULL THEN
 
      colNames := ColumnsOfSelectStmt(iSQL);
      /*
      ** If query was SELECT * or something else happened
      ** then just use the regular ColumnsOf function
      */

      /*
      ** ### PROBLEM ### if user leaves PK columns out of
      **                 select.
      */
      IF   colNames.Exists(1) AND colNames(1) = '*' THEN
        colNames := ColumnsOf(t);
      ELSE
        specifiedSelectList    := TRUE;
        DECLARE
          theSelectList VARCHAR2(2000) := SelectList(colNames);
        BEGIN
-- ** Removed uppercase function 8/4/03  S.Kho         
--	    theSelectList := UPPER(theSelectList);
          theSelectList := REPLACE(theSelectList,' ');
          theSelectList := ','||theSelectList||',';
          userSuppliedSelectList := theSelectList;
        END;
      END IF;

    ELSE

      colNames := ColumnsOf(t);

    END IF;

    BEGIN

      /*
      ** Build the Query to retrieve data for the current table,
      ** subject to WHERE clause restriction of parent/initial
      ** links.
      **
      ** If this is the first time through, and the caller
      ** specified a full SQL statement to use in the "iSQL"
      ** parameter, then use that instead of building up one
      ** ourselves. If we *do* build it up ourselves, then
      ** we use the KC and KV arrays to construct the WHERE
      ** clause.
      **
      ** If this is the first time through, the KC and KV arrays
      ** will contain the parts of the WHERE clause to add to
      ** restrict the query to the desired result.
      **
      ** If this is the 2nd to Nth time through, then the
      ** Constraint Array will have been used to drive the
      ** inbound values of KC and KV so that the detail queries
      ** are properly restricted to be the children of their
      ** current master.
      **
      */
      IF timesIntoGet = 1 AND iSQL IS NOT NULL THEN

        q := iSQL;

      ELSE

        q := 'SELECT '||selectList(colnames)||
              ' FROM '||t||' '||whereClause(kc,kv);
      END IF;

      /*
      ** If this is the top-level, then print the Query into 
      ** the output stream
      */
      IF timesIntoGet = 1 THEN

        /*
        ** For the top-level query, add the ORDER BY clause
        ** if provided
        */
        IF ord IS NOT NULL AND iSQL IS NULL THEN

          q := q || ' ORDER BY '||replace(ord,'-',' DESC');

        END IF;

        /*
        ** If the requestor asked to "OR" the where clause
        ** conditions, instead of "AND"ing them (the default)
        ** string replace AND with OR.
        */
        IF upper(cond) = 'OR' AND iSQL IS NULL THEN

          q := REPLACE(q,'   AND ','    OR ');
   
        END IF;

        WriteQueryToStream( q );

      END IF;

      /*
      ** Open Cursor
      */
      theCursor := dbms_sql.open_cursor;

      /*
      ** Parse
      */
      dbms_sql.parse (theCursor, q, dbms_sql.native);

      /*
      ** Define Columns
      */
      FOR i IN 1 .. 255 LOOP
        BEGIN
          dbms_sql.define_column( theCursor, i, columnValue, 32765);
            colCnt := colCnt + 1;
        EXCEPTION WHEN OTHERS THEN
          IF ( sqlcode = -1007) THEN
            exit;
          ELSE
            null;
          END IF;
        END;
      END LOOP;
      /*
      ** Execute Cursor
      */
      status := dbms_sql.execute(theCursor);
      maxRows := max_rows + NVL(minimum_row,1) - 1;
      /*
      ** Begin Loop
      */
      LOOP /* over the rows */
        EXIT WHEN (rowCnt >= maxRows
               OR  dbms_sql.fetch_rows(theCursor) <= 0 );
        rowCnt := rowCnt + 1;
        IF rowCnt >= NVL(minimum_row,1) THEN

          /*
          ** If there's at least one row in the
          ** result, then we need to open up
          ** the Opening Tag for the table
          */
          IF rowCnt = 1 THEN

            IF the_level = 0 THEN

               IF topLevelMultiple THEN

                /*
                ** If requestor asked to allow multiple
                ** top-level records, then to keep the
                ** result conforming to XML, we need to
                ** enclose it within an outer, single-
                ** occurrence document tag
                */
                the_level := the_level + 1;
                XMLTagFor(ite(de is NOT NULL, de,t||'LIST'), the_level);
                foundARow := TRUE;                
              END IF;

            END IF; /* the_level = 0 */

            /*
            ** Print the Name of the current table
            ** one indent level in.
            */
            the_level := the_level + 1;
            IF ( t = initialTable and usersuppliedtoptag is not null ) THEN
              XMLTagFor(userSuppliedTopTag, the_level);
            ELSE
              XMLTagFor(t, the_level);
            END IF;            
          END IF; /* rowCnt = 1 */

          IF rowCnt > 1 THEN

            /*
            ** If this is row 2 through N of the result set
            ** then we need to fill in an extra
            **
            **      :
            **  <TABLENAME>
            **      :
            **
            ** tags to keep the matching going...
            */
            IF ( t = initialTable and usersuppliedtoptag is not null ) THEN
              XMLTagFor(userSuppliedTopTag, the_level);
            ELSE
              XMLTagFor(t, the_level);
            END IF;            

          END IF;


          /*
          ** Bump the level in preparation for
          ** printing the column values
          */
          the_level     := the_level + 1;


          /*
          ** Loop over all the columns in the current table
          */
          FOR i IN 1 .. colCnt LOOP
            BEGIN
              dbms_sql.column_value( theCursor, i, columnValue );

              /*
              ** Spit out the XML Instance data with appropriate tags
              **
              ** Default the Tag names to the column name for now.
              */
              theValue := columnValue;

              tagName := colNames(i);

              /*
              ** If this column number is in the constraintArray then
              ** remember the value of this column number from
              ** the current record for use in detail queries...
              */
              UpdateConstraintJoinValues( tagName, theValue, cArr );

              /*
              **
              ** Print the tagged Instance data for the table columns
              **
              */
              IF specifiedSelectList AND t = initialTable THEN
                /*
                ** If user specified select list, then
                ** only print those columns for top-level table
                */
                IF INSTR(userSuppliedSelectList, ','||tagname||',' ) > 0 THEN 
                  XMLTagFor(tagName, the_level, content=>thevalue);
                END IF;
              ELSE
                XMLTagFor(tagName, the_level, content=>QuoteCommonEntities(thevalue));
              END IF;

            EXCEPTION WHEN OTHERS THEN
              WriteToStream('Unable to fetch column. '||SQLERRM);
            END;
          END LOOP; /* Over the Columns */

        END IF; /* Greater than the minimum number of rows */

        /*
        ** Only bother to do this is we're supposed to traverse
        ** to related tables.
        */

        IF traverseRelatedTables THEN

          /*
          **  reset the pointer into the
          **  constraint array
          */
          lastConstPtr := 0;

          /*
          ** (3) Print the Tables Referenced by the current table
          */

          LOOP

            /*
            ** Recursively print the structure of the referenced table
            ** pulling in only child rows for the current
            ** parent row.
            **
            ** This means we need to calculate a new set of values
            ** for the KEY Columns (kc) and the KEY Values (kv)
            ** to recursively into get() again
            **
            ** We need the KEY Columns to be the list of columns
            ** in the constraint that links the tables
            ** and the KEY Values to be the values from the
            ** current instance for the matching columns in
            ** the current row. get() will use those to
            ** build the right WHERE clause to restrict
            ** the rows queried to just the proper child
            ** rows.
            */
            nextConstraint( cArr,
                            lastConstPtr,
                            targetTableName,
                            oneOrMany,
                            newcolList,
                            newvalList);

            EXIT WHEN ( targetTableName IS NULL );

            /*
            ** Decrement the indent by one to keep the nested
            ** referenced structures at the same level as
            ** we are now. But don't do it if the constraint
            ** is a one to many, and the user wants us
            ** to enclose the child collections by a <XXXXLIST>
            ** tag...
            */
            IF oneorMany = 'M' AND tagGroups THEN
              /*
              ** If we're enclosing groups with a <XXXXLIST> tag
              ** then print it. No need to decrement indent since
              ** we want the embedded structures to be indented
              ** and the get() call automatically steps in one level
              **
              */
              XMLTagFor(targetTableName||'LIST',the_level);

            ELSE

              /*
              ** If we're NOT enclosing groups with a <XXXXLIST> tag
              ** then we have to decrement indent to keep the
              ** recursive structures at the *same* level as we are
              ** now since get() call automatically steps in one level
              **
              */
              the_level := the_level - 1;

            END IF;

            /*
            ** Recursively Call Get() to format subchildren
            */
            Get( targetTableName,
                 the_level,
                 kc         => newColList,
                 kv         => newValList,
                 prevConstr => constrsSoFar );

            /*
            ** Reset indent to what it was before the increment above,
            ** but only do this if we did it above for the 'M'any case.
            */
            IF oneorMany = 'M' AND tagGroups THEN
              XMLTagFor(targetTableName||'LIST',the_level,TRUE);
            ELSE

              the_level := the_level + 1;

            END IF;

          END LOOP;

        END IF; /* TraverseRelatedTables */

        /*
        ** Outdent for closing tag for 't'
        */
        the_level := the_level - 1;
        IF ( t = initialTable and usersuppliedtoptag is not null ) THEN
          XMLTagFor(userSuppliedTopTag, the_level,TRUE);
        ELSE
          XMLTagFor(t, the_level,TRUE);
        END IF;            
 
      END LOOP; /* Over the Rows */

      dbms_sql.close_cursor(theCursor);

      /*
      ** If we were printing multiple top-level
      ** rows, we need to close out the enclosing <XXXLIST>
      **
      */
      IF timesIntoGet = 1 AND topLevelMultiple THEN
        the_level := the_level - 1;
        IF NOT foundARow THEN
          XMLTagFor(ite(de is NOT NULL, de,t||'LIST'), the_level);
        END IF;
        XMLTagFor(ite(de is NOT NULL, de,t||'LIST'), the_level,TRUE);
      END IF;


    EXCEPTION WHEN OTHERS THEN
      WriteErrorToStream(sqlerrm, q );
    END;

    timesIntoGet := timesIntoGet - 1;

  END Get;

  PROCEDURE Query( theQuery            VARCHAR2,
                   theMainTable        VARCHAR2 := NULL,
                   theDocElement       VARCHAR2 := NULL,
                   tableElement        VARCHAR2 := NULL,
                   singleRow           VARCHAR2 := 'N',
                   includeDetails      VARCHAR2 := 'N',
                   detailExclusionList VARCHAR2 := NULL,
                   maximumRows         NUMBER   := 200,
                   styleSheet          VARCHAR2 := NULL,
                   MIMEType            VARCHAR2 := 'text/xml' ) IS
    theTable              VARCHAR2(80) := theMainTable;
    theSelectListStart    NATURAL;
    theSelectListEnd      NATURAL;
    comma_at              NATURAL;
    from_at               NATURAL;
    expectingMultipleRows VARCHAR2(1);
  BEGIN
    IF UPPER(singleRow) = 'N' THEN
      expectingMultipleRows := 'Y';
    ELSE
      expectingMultipleRows := 'N';
    END IF;

    IF theTable IS NULL THEN
      theTable := FirstTableOfSelectStmt(replace(theQuery,'%20',' '));
    END IF;

    Get(tb    => theTable,
        iSQL  => replace(theQuery,'%20',' '),
        de    => theDocElement,
        tt    => tableElement,
        mult  => expectingMultipleRows,
        tr    => includeDetails,
        ex    => detailExclusionList,
        maxr  => maximumRows,
        ss    => stylesheet,
        mt    => MIMEType);

  END;

END; /* Package */
.
/
show errors


spool off;
